import subprocess
import os
import re
import urllib.request
import glob
from enum import Enum
from types import SimpleNamespace

aptRepositoryPath = "/etc/apt/sources.list.d/"
lantekPackageName = "lantek4"
lantekCcmbPackageName = "lantek4-ccmb"
systemImageDir = "full-system-images/"
systemImageZipName = "lantek4-full-system-image-"
usbMountPoint = "/mnt/usb"
usbRepositoryPath = usbMountPoint + "/debs/armhf/"
usbPackageRelativePath = "./dists/trend-offline/main/binary-armhf/"
usbPackagePath = usbMountPoint + "/debs/armhf/dists/trend-offline/main/binary-armhf/"
swapEnableScript = "/opt/LanTEK/scripts/swap_memory/enable.sh"
swapDisableScript = "/opt/LanTEK/scripts/swap_memory/disable.sh"

class ReturnCode(Enum):
    NoError = 0
    Ok = 0
    AptUpdateFailed = 1
    AptPackageInfoFailed = 2
    AptDepencencyNotFound= 3
    AptPolicyFailed = 4
    UsbDriveNotFound = 5
    UsbDriveFailedToGetFreeSpace = 6
    UsbDriveNotEnoughFreeSpace = 7
    FailedToCreateRepoDir = 8
    DpkgIsBroken = 9
    AptDownloadPackageFailed = 10
    UpdateRepositoryFailed = 11
    AptInstallFailed = 12
    UnsupportedMode = 13
    UnknownError = 14
    NotEnoughSpaceToUnzipSystemImage = 15
    SystemImageUnzipFailed = 16
    SystemImageMd5Error = 17
    SystemImageNotFound = 18
    SystemImageInvalidData = 19
    SystemImageDownloadFail = 20
    SystemImageNetworkFail = 21

def validateRepository(repository):
    command = "test -f " + aptRepositoryPath + repository + ".list"

    result, output = runBashCommand(command)

    if result == False:
        return ReturnCode.UnknownError

    return ReturnCode.Ok

def runBashCommand(command):
    command = "bash -c ' " + command + " ' "

    result = True
    output = ""
    try:
        output = subprocess.check_output(command, shell=True, universal_newlines=True) #, stderr=STDOUT, timeout = 30 * 60
    except:
        result = False

    return result, output

def runDownloadBashCommand(command):
    command = "bash -c ' " + command + " ' "

    result = True
    output = ""
    errorCode = 0
    try:
        output = subprocess.check_output(command, shell=True, universal_newlines=True)
    except subprocess.CalledProcessError as err:
        result = False
        errorCode = err.returncode
        print("error code", err.returncode, err.output)

    return result, output, errorCode

def fileMd5(filePath):
    command = "md5sum " + filePath
    result, output = runBashCommand(command)
    if result == False:
        return ReturnCode.UnknownError

    return output.split(" ")[0]

def verifyDpkgCommand():
    return "dpkg --configure -a"

def aptOptionsForRepo(repository):
    return " -o Dir::Etc::sourcelist=" + aptRepositoryPath + repository + ".list -o Dir::Etc::sourceparts='-' -o APT::Get::List-Cleanup='0' "

def aptUpdateCommand(repository):
    return "apt-get update " + aptOptionsForRepo(repository)

def aptPackageDetailedInfoCommand(repository, packageName):
    return "apt-cache showpkg " +  aptOptionsForRepo(repository) + packageName

def aptPackageRawInfoCommand(repository, packageName, packageVersion):
    return "apt-cache show %s %s=%s" %  (aptOptionsForRepo(repository), packageName, packageVersion)

def aptPolicyCommand(repository, packageName):
    return "apt-cache policy" + aptOptionsForRepo(repository) + packageName

def aptDependsCommand(repository, packageName, packageVersion):
    return "apt depends %s %s=%s " % (aptOptionsForRepo(repository) , packageName , packageVersion)

def aptDownloadCommand(repository, packageName):
    return "apt-get download " + aptOptionsForRepo(repository) + packageName

def aptInstallCommand(repository, packageName, version):
    command = "apt-get  --assume-yes --allow-downgrades install " + aptOptionsForRepo(repository) + packageName
    if len(version) > 0:
        command += "=" + version
    return command

def refreshUsbRepositoryCommand():
    return "cd " + usbRepositoryPath + " && dpkg-scanpackages --multiversion " + usbPackageRelativePath + " /dev/null | gzip -9c > " + usbPackageRelativePath + "/Packages.gz && gunzip -c " + usbPackageRelativePath + "Packages.gz > " + usbPackageRelativePath + "Packages"

def dependenciesCommandPart(repository, packageName, packageVersion):
    dependencies = getDependencies(repository, packageName, packageVersion)

    if isinstance(dependencies, list) == False:
        print("get dependency version failed")
        return ReturnCode.UnknownError #in this case it's an error code

    packagesCommandPart = ""
    for dependency in dependencies:
        packagesCommandPart += " " + dependency.name + "=" + dependency.version

    #include teamviewer manually as if its included in the dependancy list above
    #it will remove lantek4 package as well
    packagesCommandPart += " teamviewer-host=15.9.5"
    return packagesCommandPart

def supportedAptRepositories():
    files = glob.glob(aptRepositoryPath+"trend-*.list")
    files = {file.replace(aptRepositoryPath, '').replace(".list","") for file in files}

    files = sorted(files)

    print("Repositories:", list(files))

    return ReturnCode.Ok

def getRemoteFileSize(url):
    req = urllib.request.Request(url, method='HEAD')
    file = urllib.request.urlopen(req)
    if file.status != 200:
        return -1

    return file.headers['Content-Length']

def getRemoteFileContent(url):
    return urllib.request.urlopen(url).read()

def getRepositoryUrl(repository):
    command = "cat %s/%s.list" % (aptRepositoryPath, repository)
    result, output = runBashCommand(command)

    if result == False:
        return ReturnCode.UnknownError

    urlLine = ""
    lines = output.split("\n")
    for line in lines:
        if line.startswith("#") or "deb" not in line :
            continue

        urlLine = line
        break

    if urlLine == "":
        return ReturnCode.UnknownError

    parts = urlLine.split(" ")
    urlLine = ""

    for part in parts:
        if "http://" not in part:
            continue
        urlLine = part

    if urlLine == "":
        return ReturnCode.UnknownError

    urlLine += "dists/%s/main/binary-armhf/" % repository

#    print("url:", urlLine)

    return urlLine

def systemImageUrl(repository, fileName):
    url = getRepositoryUrl(repository)
    if url == ReturnCode.UnknownError:
        return url

    url += systemImageDir + fileName

    return url

def systemImagesInfoUrl(repository):
    url = getRepositoryUrl(repository)
    if url == ReturnCode.UnknownError:
        return url

    url += systemImageDir + "Images"

    return url

def getZipUncompressedSize(fileName):
    command = "unzip -Zt " + fileName

    result, output = runBashCommand(command)
    if result == False:
        print("unzip -Zt command failed")
        return ReturnCode.UnknownError

    parts = output.split(", ")
    if len(parts) != 3:
        print("unexpected output: expected 3 parts")
        return ReturnCode.UnknownError

    uncompressedSize = parts[1].split(" ")[0]
    if uncompressedSize == "":
        print("unexpected empty uncompressedSize")
        return ReturnCode.UnknownError

    return uncompressedSize

def getAptPolicyOutput(repository, packageName):
    command = aptPolicyCommand(repository, packageName)
    result, output = runBashCommand(command)

#    print output
#    print exitCode

    if result == False:
        print("apt policy failed")
        return ReturnCode.AptPolicyFailed

    lines = output.split("\n")

    return lines

def getAptDependsForPackage(repository, packageName, packageVersion):
    command = aptDependsCommand(repository, packageName, packageVersion)
    result, output = runBashCommand(command)

#    print output
#    print exitCode

    if result == False:
        print("apt policy failed")
        return ReturnCode.AptPolicyFailed

    lines = output.split("\n")

    return lines

def usbDriveConnected():
    command = "findmnt " + usbMountPoint
    result, output = runBashCommand(command)
    if result == False:
        print("USB drive not found")
        return ReturnCode.UsbDriveNotFound

    return ReturnCode.Ok

# requiredSpace: in kB
def validateUsbFreeSpace(requiredSpace):
    command = "df | grep " + usbMountPoint
    result, output = runBashCommand(command)
    if result == False or output == "":
        print("getting free space for USB drive failed")
        return ReturnCode.UsbDriveFailedToGetFreeSpace

    parts = output.split()
    if len(parts) != 6:
        print("unexpected output from getFreeSpace command")
        return ReturnCode.UsbDriveFailedToGetFreeSpace

    freeSpace = int(parts[3])
    print("there's " + str(freeSpace) + " of free space on USB")

    minumumRequiredUsbFreeSpace = requiredSpace / 1024 #freeSpace comes in kilobytes

#    print("freeSpace:", freeSpace)
#    print("requiredSpace:", requiredSpace)
    if freeSpace < minumumRequiredUsbFreeSpace:
        print("there's no enough space on USB drive")
        return ReturnCode.UsbDriveNotEnoughFreeSpace

    return ReturnCode.Ok

def getDependencies(repository, packageName, packageVersion):
    command = aptPackageDetailedInfoCommand(repository, packageName)
    result, output = runBashCommand(command)

#    print("command:", command)

    if result == False:
        print("apt package info failed")
        return ReturnCode.AptPackageInfoFailed

    lines = output.split("\n")

    dependencyLine = ""
    isDependencyLine = False
    for line in lines:
        if "Dependencies:" in line:
            isDependencyLine = True
            continue
        if not isDependencyLine:
            continue

#        print("line: " + line)

        if "Provides:" in line:
            break

        if not packageVersion:
            dependencyLine = line
            break

        if line.startswith(packageVersion) :
            dependencyLine = line
            break

    if dependencyLine == "":
        return []

#    print("dependency:", dependencyLine)

    parts = dependencyLine.split(") ")

    dependencies = []

    for part in parts:
        part = part.strip()

        if "0 (null)" in part:
            continue

        subparts = part.split(" ")

#        print("subparts:", subparts)
        if len(subparts) != 3:
            continue

        dependency = SimpleNamespace()
        dependency.name = subparts[0]
        dependency.version = subparts[2]
        dependencies.append(dependency)

    return dependencies

def getDownloadSize(repository, packageName, packageVersion):
    dependencies = getDependencies(repository, packageName, packageVersion)

    if isinstance(dependencies, list) == False:
        print("get dependency version failed")
        return ReturnCode.UnknownError #in this case it's an error code

    downloadSize = 0

    dependency = SimpleNamespace()
    dependency.name = packageName
    dependency.version = packageVersion
    dependencies.append(dependency)

    for dependency in dependencies:
        command = aptPackageRawInfoCommand(repository, dependency.name, dependency.version)
        result, output = runBashCommand(command)
#        print("command:", command)

        if result == False:
            print("apt dependency not found dd")
            return ReturnCode.AptDepencencyNotFound

        lines = output.split("\n")

        dependencySize = 0
        for line in lines:
            if "Size" not in line:
                continue

            parts = line.split(" ")
            dependencySize = int(parts[1])

#            print("size:", dependencySize)

        if dependencySize == 0:
#            print("skipping %s dependency" % (dependency.name))
            continue

        downloadSize += dependencySize

    return downloadSize

def aptDownloadSize(repository, packageName, packageVersion):
    command = aptUpdateCommand(repository)
    result, output = runBashCommand(command)

    if result == False:
        print("apt update failed")
        return ReturnCode.AptUpdateFailed

    downloadSize = getDownloadSize(repository, packageName, packageVersion)

    if downloadSize == ReturnCode.UnknownError or downloadSize == ReturnCode.AptDepencencyNotFound:
        return downloadSize

    print("DownloadSize:", downloadSize)
    return ReturnCode.Ok

def systemImageDownloadSize(repository, imageName, version):
    #get image info
    imageInfo = downloadSystemImageInfo(repository, imageName, version)
    if imageInfo == ReturnCode.UnknownError:
        return ReturnCode.UnknownError

    if imageInfo.name == "" and imageInfo.version == "" and imageInfo.fileName == "" and  imageInfo.fileSize == "" and  imageInfo.md5 == "":
        return ReturnCode.SystemImageNotFound

    if imageInfo.name == "" or imageInfo.version == "" or imageInfo.fileName == "" or  imageInfo.fileSize == "" or  imageInfo.md5 == "":
        return ReturnCode.SystemImageInvalidData

    print("DownloadSize:", imageInfo.fileSize)

    return ReturnCode.Ok

def downloadSystemImageRepositoryInfo(repository, name):
    url = systemImagesInfoUrl(repository)
    if url == ReturnCode.UnknownError:
        return ReturnCode.UnknownError

    imagesData = getRemoteFileContent(url).decode("utf-8")

    lines = imagesData.split("\n")
    images = []
    imageInfo = SimpleNamespace()
    for line in lines:
        if line.startswith("Image: "):
            imageInfo.name = line.replace("Image: ", "")
        if line.startswith("Version: "):
            imageInfo.version = line.replace("Version: ", "")
        if line.startswith("Filename: "):
            imageInfo.fileName = line.replace("Filename: ", "")
        if line.startswith("Size: "):
            imageInfo.fileSize = line.replace("Size: ", "")
        if line.startswith("MD5sum"):
            imageInfo.md5 = line.replace("MD5sum: ", "")
        if line == "":
            if not hasattr(imageInfo, "name") or imageInfo.name == "" or imageInfo.version == "" or imageInfo.fileName == "" or imageInfo.fileSize == "" or imageInfo.md5 == "":
                continue

            images.append(imageInfo)
            imageInfo = SimpleNamespace()

    return images

def downloadSystemImageInfo(repository, name, version):
    candidates = downloadSystemImageRepositoryInfo(repository, name)
    if isinstance(candidates, list) == False:
        return ReturnCode.UnknownError

    for candidate in candidates:
        if candidate.name.lower() == name.lower() and candidate.version == version:
            return candidate

    return ReturnCode.UnknownError

def downloadSystemImage(repository, imageName, version):
    #get image info
    imageInfo = downloadSystemImageInfo(repository, imageName, version)
    if imageInfo == ReturnCode.UnknownError:
        return ReturnCode.UnknownError

    if imageInfo.name == "" and imageInfo.version == "" and imageInfo.fileName == "" and  imageInfo.fileSize == "" and  imageInfo.md5 == "":
        return ReturnCode.SystemImageNotFound

    if imageInfo.name == "" or imageInfo.version == "" or imageInfo.fileName == "" or  imageInfo.fileSize == "" or  imageInfo.md5 == "":
        return ReturnCode.SystemImageInvalidData

    #get image URL
    url = systemImageUrl(repository, imageInfo.fileName)
    if url == ReturnCode.UnknownError:
        return url

    #check if USB drive is mounted
    result = usbDriveConnected()
    if result != ReturnCode.Ok:
        return result

    #erase USB drive
    command = "rm -fr %s/*" % usbMountPoint
    result, output = runBashCommand(command)

    downloadSize = imageInfo.fileSize

    #check if USB drive has enough space
    result = validateUsbFreeSpace(int(downloadSize))
    if result != ReturnCode.Ok:
        return result

    targetDir = usbPackagePath + systemImageDir
    systemImageFilePath = targetDir + imageInfo.fileName
    #remove file if exists
    command = "rm -f " + systemImageFilePath
    result, output = runBashCommand(command)
    #default retries is 20, setting it to 3, wait for 20 secs max if no data is recieved and try 5 times
    command = "wget --backups=0 --timeout=20 --tries=5 %s -P %s" % (url, targetDir)
#    print("command:", command)
    result, output, errorCode = runDownloadBashCommand(command)
    if result == False:
        print("download failed")
        print(output)
        if errorCode == 4:
            return ReturnCode.SystemImageNetworkFail
        return ReturnCode.SystemImageDownloadFail

    #check md5sum
    md5sum = fileMd5(systemImageFilePath)
#    print("downloaded MD5:", md5sum, "expected MD5:", imageInfo.md5)
    if md5sum != imageInfo.md5:
        print("MD5 check failed")
        return ReturnCode.SystemImageMd5Error

    #check zip uncompressed size
    unzippedSize = getZipUncompressedSize(systemImageFilePath)

    if unzippedSize == ReturnCode.UnknownError:
        return ReturnCode.UnknownError

#    print("unzippedSize", unzippedSize)
    #check if USB drive has enough space now
    result = validateUsbFreeSpace(int(unzippedSize))
    if result != ReturnCode.Ok:
        print("insufficient free space to unzip system image")
        return ReturnCode.NotEnoughSpaceToUnzipSystemImage

    # unzip
    command = "cd %s && unzip -o %s" % (usbMountPoint, systemImageFilePath)
    result, output = runBashCommand(command)
    if result == False:
        print("unzip failed")
        return ReturnCode.SystemImageUnzipFailed

    return ReturnCode.Ok


def systemImageCandidates(repository, packageName):
    candidates = downloadSystemImageRepositoryInfo(repository, packageName)
    if isinstance(candidates, list) == False:
        return ReturnCode.UnknownError

    versions = []
    for candidate in candidates:
        versions.append(candidate.version)

    print("Candidates:", versions)

    return ReturnCode.Ok

def checkIfRemoteFileExists(url):
    try:
        r = urllib.request.urlopen(url)
        return r.getcode() == 200
    except urllib.request.HTTPError:
        return False


def downloadUserNotice(repository, candidateVersion):
    imagesData = ""

    if repository == "trend-offline":
        filePath = usbPackagePath
        filePath += candidateVersion+ "-LTIV-UserNotice.txt"
        fileOk = os.path.isfile(filePath)

        if fileOk == True:
           with open(filePath) as f:
                imagesData = f.read()
    else:
        url = getRepositoryUrl(repository)
        if url == ReturnCode.UnknownError:
            return ReturnCode.UnknownError
        url += candidateVersion+ "-LTIV-UserNotice.txt"
        fileOk = checkIfRemoteFileExists(url)
        if fileOk == False:
            return imagesData
        imagesData = getRemoteFileContent(url).decode("utf-8")

    return imagesData

def aptGetUserNotice(repository, packageName, packageVersion):
    noticeText = downloadUserNotice(repository, packageVersion)
    if noticeText == "":
        return ReturnCode.Ok
    else:
        noticeText += "\n"
    print(noticeText)
    return ReturnCode.Ok

def aptCheckUpdate(repository, packageName):
    #apt update
    command = aptUpdateCommand(repository)
    result, output = runBashCommand(command)

    print(command)
    print(output)
    if result == False:
        print("apt update failed")
        return ReturnCode.AptUpdateFailed

    lines = getAptPolicyOutput(repository, packageName)

    if lines == ReturnCode.AptPolicyFailed or lines == ReturnCode.AptUpdateFailed:
        return lines

    for line in lines:
        if "Installed:" in line:
            installedVersion = line
        elif "Candidate:" in line:
            candidateVersion = line

    installedVersion = installedVersion.replace(" ", "")
    candidateVersion = candidateVersion.replace(" ", "")

    print(installedVersion)
    print(candidateVersion)

    return ReturnCode.Ok

def aptFetchCCMBVersion(repository, packageName, packageVersion):
    #apt update
    command = aptUpdateCommand(repository)
    result, output = runBashCommand(command)

    print(command)
    print(output)
    if result == False:
        print("apt update failed")
        return ReturnCode.AptUpdateFailed

    lines = getAptDependsForPackage(repository, packageName,packageVersion)

    if lines == ReturnCode.AptPolicyFailed or lines == ReturnCode.AptUpdateFailed:
        return lines

    for line in lines:
        if "lantek4-ccmb" in line:
            ccmbVersion = line.partition("=")[2]

    ccmbVersion = ccmbVersion.replace(")", "")
    ccmbVersion = "lantek4-ccmb:" + ccmbVersion
    print(ccmbVersion)
    return ReturnCode.Ok

def aptDownloadUpdate(repository, packageName, packageVersion):
    #apt update
    command = aptUpdateCommand(repository)
    result, output = runBashCommand(command)

    if result == False:
        print("apt update failed")
        return ReturnCode.AptUpdateFailed

    #check if USB drive is mounted
    result = usbDriveConnected()
    if result != ReturnCode.Ok:
        return result

    #check if USB drive has enough space
    downloadSize = getDownloadSize(repository, packageName, packageVersion)

    if downloadSize == ReturnCode.UnknownError or downloadSize == ReturnCode.AptDepencencyNotFound:
        return downloadSize

    result = validateUsbFreeSpace(downloadSize)
    if result != ReturnCode.Ok:
        return result

    #make repo dir on USB
    command = "mkdir -p " + usbPackagePath
    result, output = runBashCommand(command)
    if result == False:
        print("creating repo dir failed")
        return ReturnCode.FailedToCreateRepoDir

    #clean up repo dir (if it existed before)
    command = "rm -fr " + usbPackagePath + "/*"
    result, output = runBashCommand(command)
    if result == False:
        print("cleaning up repo dir failed")
        return ReturnCode.FailedToCreateRepoDir

    packagesToDownload = packageName
    if packageVersion:
        packagesToDownload = packageName + "=" + packageVersion

    #dependencies
    dependenciesPart = dependenciesCommandPart(repository, packageName, packageVersion)
    if dependenciesPart == ReturnCode.UnknownError:
        return ReturnCode.UnknownError

    packagesToDownload += " " + dependenciesPart

    #download packages
    command = aptDownloadCommand(repository, packagesToDownload)

    command = "cd " + usbPackagePath + " && " + command

    result, output = runBashCommand(command)
    if result == False:
        print("download packages failed")
        return ReturnCode.AptDownloadPackageFailed

    #update usb repo
    command = refreshUsbRepositoryCommand()

    result, output = runBashCommand(command)
    if result == False:
        print("update repo failed")
        return ReturnCode.UpdateRepositoryFailed

    return ReturnCode.Ok

def aptInstallUpdate(repository, packageName, packageVersion):
    #apt update
    command = aptUpdateCommand(repository)
    result, output = runBashCommand(command)

    if result == False:
        print("apt update failed")
        return ReturnCode.AptUpdateFailed

    command = aptInstallCommand(repository, packageName, packageVersion)

    #add dependencies to install command
    dependenciesPart = dependenciesCommandPart(repository, packageName, packageVersion)
    if dependenciesPart == ReturnCode.UnknownError:
        return ReturnCode.UnknownError

    command += " " + dependenciesPart

#    print("command:", command)
    runBashCommand(swapEnableScript)

    result, output = runBashCommand(command)

    runBashCommand(swapDisableScript)

    if result == False:
        print("package installation failed:", command)
        return ReturnCode.AptInstallFailed

    return ReturnCode.Ok

def aptPackageCandidates(repository, packageName):
    #apt update
    command = aptUpdateCommand(repository)
    result, output = runBashCommand(command)

    print(command)
    if result == False:
        print("apt update failed")
        return ReturnCode.AptUpdateFailed

    lines = getAptPolicyOutput(repository, packageName)

    if lines == ReturnCode.AptPolicyFailed or lines == ReturnCode.AptUpdateFailed:
        return lines

    candidates = []
    previousLine = ""
    for line in lines:
        #500 http is accessible in repo
        #*** is installed package
        if "500 http" in line and repository in line and "***" not in line:
            previousLine = previousLine[5 :  :  ] #remove first 5 chars
            parts = previousLine.split(" ")
            candidates.append(parts[0])
        previousLine = line

    print("Candidates:", candidates)
    return ReturnCode.Ok

